/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.
	
	pgpDriverNT.c - kernel mode device driver for PGP utility functions
	

	$Id: pgpDriverNT.c,v 1.15 2001/03/12 21:24:40 pbj Exp $
____________________________________________________________________________*/

#include "ntddk.h"
#include "stdarg.h"

#ifndef PGP_EROS
#define PGP_EROS	0
#endif // PGP_EROS

#if !PGP_EROS
#include "ntddkbd.h"
#include "ntddmou.h"
#endif // !PGP_EROS

#include "PGPsdkDriver.h"

#include "pgpDriverNT.h"
#include "pgpMemlockNT.h"
#include "pgpGeneric.h"
#include "pgpMisc.h"

#if !PGP_EROS
#include "pgpWipeDeleteNT.h"
#include "pgpEntropy.h"

static WCHAR	keyboardGuid[]	= L"{4D36E96B-E325-11CE-BFC1-08002BE10318}";
static WCHAR	mouseGuid[]		= L"{4D36E96F-E325-11CE-BFC1-08002BE10318}";
#endif // !PGP_EROS

static WCHAR	deviceNameBuffer[]	= L"\\Device\\PGPsdkDriver";
static WCHAR	deviceLinkBuffer[]  = L"\\DosDevices\\PGPsdkDriver";


#if !PGP_EROS
//	______________________________________________________
//
//	I/O Completion routine for mouse read requests
//
//	this routine is called after the system has retrieved the
//  mouse position and button information from the lower level driver.
//
//	if a mouse button has been pressed, we need to reset the inactivity 
//  timers.  Also, if count is reached, we pass timing information into 
//	the entropy hashing functions

static NTSTATUS
sMouseReadCompletion (
	IN PDEVICE_OBJECT	pdevo,
	IN PIRP				pirp,
	IN PVOID			context)
{
	PDEVEXTENSION 	pdeveFilter		= pdevo->DeviceExtension;
	PDRVEXTENSION 	pdrve			= pdeveFilter->pdrve;

	PGPdbgVerbosePrint(( "PGPutil: sReadCompletion: IRP: %x\n", pirp));

	// if appropriate, send this IRP to the entropy store driver
	if (NT_SUCCESS(pirp->IoStatus.Status))
	{
		PMOUSE_INPUT_DATA	pmid;
		ULONG				ulForceAdd;

		// hash the x, y coordinates into the entropy buffer
		pmid = (PMOUSE_INPUT_DATA)(pirp->AssociatedIrp.SystemBuffer);

		// flag that the hook seems to be working
		pdrve->ulStatusFlags |= kPGPUDFlag_MouseHookCalled;

		if (pdeveFilter->ulSkipCount == 0)
		{
			pdeveFilter->ulSkipCount = MOUSE_SKIP;
			ulForceAdd = TRUE;
		}
		else
		{
			pdeveFilter->ulSkipCount--;
			ulForceAdd = FALSE;
		}

		// add entropy to pool
		// (leave as SPINLOCK we could be > dispatch level)
		if (pgpDriverTryToEnterCriticalSection (&pdrve->csEntropy, 
			SPINLOCK))
		{
			pgpEntropyAddMouseEvent (ulForceAdd);
			pgpDriverLeaveCriticalSection (&pdrve->csEntropy, SPINLOCK);
		}

		if (pmid->Buttons != 0)
		{
			pgpProcessActivityEvent (&pdrve->inactivity,
									&pdrve->csInactivity);
		}
	}

	if (pirp->PendingReturned)
	{
		IoMarkIrpPending (pirp);
	}
	return STATUS_SUCCESS;
}


//	______________________________________________________
//
//  process the IRPs sent to the mouse device.
//
//	this function simply passes a request on to the lower driver,
//	attaching a completion routine pointer to the "read" requests.

static NTSTATUS
sMouseEntropyDispatch (
	IN PDEVICE_OBJECT	pdevo,
	IN PIRP				pirp)
{
	PDEVEXTENSION	 	pdeveFilter		= pdevo->DeviceExtension;
	PIO_STACK_LOCATION 	pirpsCurrent	= IoGetCurrentIrpStackLocation (pirp);

	PGPdbgVerbosePrint (("PGPutil: sKbdEntropyDispatch: IRP: %8x; "
					"MajorFunction: %d\n", pirp,
					pirpsCurrent->MajorFunction));

	switch (pirpsCurrent->MajorFunction) {
	case IRP_MJ_READ :
		// Copy args to next level
		IoCopyCurrentIrpStackLocationToNext(pirp);

		// Set up Completion routine to handle marking the IRP pending.
		IoSetCompletionRoutine (pirp, sMouseReadCompletion, 
								NULL, TRUE, TRUE, TRUE);

		// Pass the IRP to the target driver
		return IoCallDriver (pdeveFilter->pdevoNext, pirp);

#if	(_WIN32_WINNT >= 0x0500)

	case IRP_MJ_PNP:
	    switch (pirpsCurrent->MinorFunction)
		{
	    case IRP_MN_REMOVE_DEVICE:        
			// target device is going away, so do we
			IoSkipCurrentIrpStackLocation(pirp);
			IoCallDriver(pdeveFilter->pdevoNext, pirp);

			IoDetachDevice(pdeveFilter->pdevoNext); 
			IoDeleteDevice(pdevo);

			return STATUS_SUCCESS;

		default:
			// Pass the IRP to the target
			IoSkipCurrentIrpStackLocation(pirp);
			return IoCallDriver (pdeveFilter->pdevoNext, pirp);
		}

	case IRP_MJ_POWER:
		// must use special calls to pass down power IRPs
		PoStartNextPowerIrp(pirp);
		IoSkipCurrentIrpStackLocation(pirp);

		return PoCallDriver(pdeveFilter->pdevoNext, pirp);

#endif	// _WIN32_WINNT >= 0x5000

	default :
		// Pass the IRP to the target
		IoSkipCurrentIrpStackLocation(pirp);
		return IoCallDriver (pdeveFilter->pdevoNext, pirp);
	}
}


//	______________________________________________________
//
//	I/O Completion routine for keyboard read requests
//
//	this routine is called after the system has retrieved the
//  keypress information from the lower level driver.
//
//	we need to reset the inactivity timers, and if count is 
//	reached, pass keypress into the entropy hashing functions

static NTSTATUS
sKbdReadCompletion (
	IN PDEVICE_OBJECT	pdevo,
	IN PIRP				pirp,
	IN PVOID			context)
{
	PDEVEXTENSION 	pdeveFilter		= pdevo->DeviceExtension;
	PDRVEXTENSION 	pdrve			= pdeveFilter->pdrve;

	PGPdbgVerbosePrint(( "PGPutil: sReadCompletion: IRP: %x\n", pirp));

	// if appropriate, send this IRP to the entropy store driver
	if (NT_SUCCESS(pirp->IoStatus.Status))
	{
		PKEYBOARD_INPUT_DATA	pkid;

		// put the scancode in the structure to pass to the
		// Entropy driver callback function
		pkid = (PKEYBOARD_INPUT_DATA)(pirp->AssociatedIrp.SystemBuffer);

		// only take key presses
		if (pkid->Flags == KEY_MAKE) 
		{
			ULONG	ulForceAdd;

			// flag that the hook seems to be working
			pdrve->ulStatusFlags |= kPGPUDFlag_KeyboardHookCalled;

			if (pdeveFilter->ulSkipCount == 0)
			{
				pdeveFilter->ulSkipCount = KEYBOARD_SKIP;
				ulForceAdd = TRUE;
			}
			else
			{
				pdeveFilter->ulSkipCount--;
				ulForceAdd = FALSE;
			}

			// hash the keycode into the entropy buffer
			// (leave as SPINLOCK we could be > dispatch level)
			if (pgpDriverTryToEnterCriticalSection (&pdrve->csEntropy, 
				SPINLOCK))
			{
				pgpEntropyAddKeystrokeEvent (ulForceAdd, pkid->MakeCode);
				pgpDriverLeaveCriticalSection (&pdrve->csEntropy, 
					SPINLOCK);
			}

			pgpProcessActivityEvent (&pdrve->inactivity,
									&pdrve->csInactivity);
		}
	}

	if (pirp->PendingReturned)
	{
		IoMarkIrpPending (pirp);
	}
	return STATUS_SUCCESS;
}


//	______________________________________________________
//
//  process the IRPs sent to the keyboard device.
//
//	this function simply passes a request on to the lower driver,
//	attaching a completion routine pointer to the "read" requests.

static NTSTATUS
sKbdEntropyDispatch (
	IN PDEVICE_OBJECT	pdevo,
	IN PIRP				pirp)
{
	PDEVEXTENSION	 	pdeveFilter		= pdevo->DeviceExtension;
	PIO_STACK_LOCATION 	pirpsCurrent	= IoGetCurrentIrpStackLocation (pirp);

	PGPdbgVerbosePrint (("PGPutil: sKbdEntropyDispatch: IRP: %8x; "
					"MajorFunction: %d\n", pirp,
					pirpsCurrent->MajorFunction));

	switch (pirpsCurrent->MajorFunction) {
	case IRP_MJ_READ :
		IoCopyCurrentIrpStackLocationToNext(pirp);

		// Set up Completion routine to handle marking the IRP pending.
		IoSetCompletionRoutine (pirp, sKbdReadCompletion, 
								NULL, TRUE, TRUE, TRUE);

		// Pass the IRP to the target driver
		return IoCallDriver (pdeveFilter->pdevoNext, pirp);

#if	(_WIN32_WINNT >= 0x0500)

	case IRP_MJ_PNP:
	    switch (pirpsCurrent->MinorFunction)
		{
	    case IRP_MN_REMOVE_DEVICE:        
			// target device is going away, so do we
			IoSkipCurrentIrpStackLocation(pirp);
			IoCallDriver(pdeveFilter->pdevoNext, pirp);

			IoDetachDevice(pdeveFilter->pdevoNext); 
			IoDeleteDevice(pdevo);

			return STATUS_SUCCESS;

		default:
			// Pass the IRP to the target
			IoSkipCurrentIrpStackLocation(pirp);
			return IoCallDriver (pdeveFilter->pdevoNext, pirp);
		}

	case IRP_MJ_POWER:
		// must use special calls to pass down power IRPs
		PoStartNextPowerIrp(pirp);
		IoSkipCurrentIrpStackLocation(pirp);

		return PoCallDriver(pdeveFilter->pdevoNext, pirp);

#endif	// _WIN32_WINNT >= 0x5000

	default :
		// Pass the IRP to the target
		IoSkipCurrentIrpStackLocation(pirp);
		return IoCallDriver (pdeveFilter->pdevoNext, pirp);
	}
}
#endif // !PGP_EROS


//	______________________________________________________
//
//  just make sure that the buffer is big enough to hold the data

static ULONG
sValidateDeviceIOBuffer (
    IN PIRP					pirp,
	IN PIO_STACK_LOCATION	pIrpStack,
	IN ULONG				ulStructSize)
{
    ULONG				ulBufLenIn			= 0;
    ULONG				ulBufLenOut			= 0;

	ulBufLenIn  = pIrpStack->Parameters.DeviceIoControl.InputBufferLength;
	ulBufLenOut = pIrpStack->Parameters.DeviceIoControl.OutputBufferLength;

	pirp->IoStatus.Status = STATUS_INVALID_PARAMETER;
	pirp->IoStatus.Information = 0;

	if (ulBufLenIn < ulStructSize) 
	{
		PGPdbgPrint (("PGPutil: sDispatchDeviceControl: "
					  	"invalid input buffer length.\n"));
		return FALSE;
	}

	if (pirp->AssociatedIrp.SystemBuffer == NULL)
	{
		PGPdbgPrint (("PGPutil: sDispatchDeviceControl: "
					  	"invalid buffer address.\n"));
		return FALSE;
	}

	pirp->IoStatus.Status = STATUS_SUCCESS;
	pirp->IoStatus.Information = ulBufLenOut;
	return TRUE;
}


//	______________________________________________________
//
//  process the IRPs sent to the "main" device (as opposed to
//	the keyboard and mouse "filter" devices.

static NTSTATUS
sPGPUtilDispatch (
    IN PDEVICE_OBJECT	pdevo,
    IN PIRP				pirp)
{
    PIO_STACK_LOCATION	pIrpStack			= NULL;
    NTSTATUS			ntStatus			= STATUS_SUCCESS;
	PVOID				pStruct				= NULL;
	PDEVEXTENSION		pdeve				= pdevo->DeviceExtension;
	PDRVEXTENSION		pdrve				= pdeve->pdrve;

	// Init to default settings
	pirp->IoStatus.Status      = STATUS_SUCCESS;
	pirp->IoStatus.Information = 0;

	// Get a pointer to the current location in the Irp. This is where
	//     the function codes and parameters are located.
	pIrpStack = IoGetCurrentIrpStackLocation (pirp);

	switch (pIrpStack->MajorFunction) {

	// called when user app calls CreateFile
	case IRP_MJ_CREATE:
		PGPdbgVerbosePrint (("PGPutil: IRP_MJ_CREATE\n"));
		break;

	// called when user app closes handle to driver --
	//	make sure all locked memory belonging to this handle gets unlocked
	case IRP_MJ_CLOSE:
		PGPdbgVerbosePrint (("PGPutil: IRP_MJ_CLOSE\n"));
		if (KeGetCurrentIrql () < DISPATCH_LEVEL)
		{
			pgpMemlockCleanupHandle (&pdrve->memlock,
						pIrpStack->FileObject, &pdrve->csMemlock);
		}
		break;

#if !PGP_EROS
	// called when other driver sends IRP
	case IRP_MJ_INTERNAL_DEVICE_CONTROL:
		PGPdbgVerbosePrint (("PGPutil: IRP_MJ_INTERNAL_DEVICE_CONTROL\n"));

		pStruct = pirp->AssociatedIrp.SystemBuffer;

		switch (pIrpStack->Parameters.DeviceIoControl.IoControlCode) {
		case IOCTL_PGPUTIL_INACTIVITY :
			if (sValidateDeviceIOBuffer (
						pirp, pIrpStack, sizeof(PGPINACTIVITYSTRUCT)))
			{
				pgpInactivityProcessOperation (
						&pdrve->inactivity,
						(PPGPINACTIVITYSTRUCT)pStruct, 
						&pdrve->csInactivity);
			}
			break;

		default:
			pirp->IoStatus.Status = STATUS_INVALID_PARAMETER;
			PGPdbgPrint (
				("PGPutil: Err: unknown IRP_MJ_INTERNAL_DEVICE_CONTROL.\n"));
			break;
		}

		break;
#endif // !PGP_EROS

	// called when user apps call DeviceIoControl
	case IRP_MJ_DEVICE_CONTROL:
		PGPdbgVerbosePrint (("PGPutil: IRP_MJ_DEVICE_CONTROL\n"));

		pStruct = pirp->AssociatedIrp.SystemBuffer;

		switch (pIrpStack->Parameters.DeviceIoControl.IoControlCode) {

		case IOCTL_PGPUTIL_GENERIC :
			if (sValidateDeviceIOBuffer (
							pirp, pIrpStack, sizeof(PPGPGENERICSTRUCT)))
			{
				pgpGenericProcessOperation (
						(PPGPGENERICSTRUCT)pStruct, pdrve->ulStatusFlags);
			}
			break;

		case IOCTL_PGPUTIL_MEMLOCK :
			if (sValidateDeviceIOBuffer (
						pirp, pIrpStack, sizeof(PGPMEMLOCKSTRUCT)))
			{
				pgpMemlockProcessOperation (
						&pdrve->memlock,
						(PPGPMEMLOCKSTRUCT)pStruct,
						pIrpStack->FileObject,
						pdrve->ulStatusFlags,
						&pdrve->csMemlock);
			}
			break;

#if !PGP_EROS
		case IOCTL_PGPUTIL_ENTROPY :
			if (sValidateDeviceIOBuffer (
							pirp, pIrpStack, sizeof(PGPENTROPYSTRUCT)))
			{
				pgpEntropyProcessOperation (
						(PPGPENTROPYSTRUCT)pStruct, &pdrve->csEntropy);
			}
			break;

		case IOCTL_PGPUTIL_CACHE :
			if (sValidateDeviceIOBuffer (
							pirp, pIrpStack, sizeof(PGPCACHESTRUCT)))
			{
				PPGPCACHE ppc	= NULL;

				switch (((PPGPCACHESTRUCT)pStruct)->ulCache) {
				case kPGPUDCache_Signing :
					ppc = &pdrve->cacheSign;
					break;
				case kPGPUDCache_Decryption :
					ppc = &pdrve->cacheDecrypt;
					break;
				default :
					((PPGPCACHESTRUCT)pStruct)->ulError = 
										kPGPUDError_BadParams;
					break;
				}

				if (ppc)
				{
					pgpCacheProcessOperation (
							ppc,
							(PPGPCACHESTRUCT)pStruct, 
							&pdrve->csCache);
				}
			}
			break;

		case IOCTL_PGPUTIL_WIPEDELETE :
			if (sValidateDeviceIOBuffer (
						pirp, pIrpStack, sizeof(PGPWIPEDELETESTRUCT)))
			{
				pgpWipeDeleteProcessOperation (
						(PPGPWIPEDELETESTRUCT)pStruct, pdrve->ulStatusFlags);
			}
			break;
#endif // !PGP_EROS

		default:
			pirp->IoStatus.Status = STATUS_INVALID_PARAMETER;
			PGPdbgPrint (
					("PGPutil: Err: unknown IRP_MJ_DEVICE_CONTROL.\n"));
			break;
		}

		break;
	}

	ntStatus = pirp->IoStatus.Status;
	IoCompleteRequest(pirp, IO_NO_INCREMENT);

	return ntStatus;
}


//	______________________________________________________
//
//  process the IRPs sent to this driver

static NTSTATUS
sDispatch (
    IN PDEVICE_OBJECT	pdevo,
    IN PIRP				pIrp)
{
	PDEVEXTENSION	pdeve	= pdevo->DeviceExtension;

	switch (pdeve->ulDeviceType) {
	case PGPUTIL_DEV :
		return sPGPUtilDispatch (pdevo, pIrp);

#if !PGP_EROS
	case KBD_ENTROPY_DEV :
		return sKbdEntropyDispatch (pdevo, pIrp);

	case MOUSE_ENTROPY_DEV :
		return sMouseEntropyDispatch (pdevo, pIrp);

	case WIPE_DELETE_DEV :
		return pgpWipeDeleteDispatch (pdevo, pIrp);
#endif // !PGP_EROS

	default :
		return STATUS_NO_SUCH_DEVICE;
    }
}


//	______________________________________________________
//
//  delete the associated devices
//	since we don't currently support unloading the driver, 
//	this routine is called only if there is a fatal error
//	during driver initialization.

static VOID
sUnload(
    IN PDRIVER_OBJECT pdrvo)
{ 
	PDEVICE_OBJECT	pdevo		= NULL;
	PDEVICE_OBJECT	pdevoNext	= NULL;
	PDEVICE_OBJECT	pgpUtilDev	= NULL;
	UNICODE_STRING	deviceLinkUniString;
	PDEVEXTENSION	pdeve;

	// get first device object
	pdevo = pdrvo->DeviceObject;

	// first get rid of all filter driver device objects
	while (pdevo)
	{
		pdevoNext = pdevo->NextDevice;
		pdeve = pdevo->DeviceExtension;

		switch (pdeve->ulDeviceType)
		{

#if !PGP_EROS
		case MOUSE_ENTROPY_DEV:
		case KBD_ENTROPY_DEV:
		case WIPE_DELETE_DEV:
			IoDetachDevice(pdeve->pdevoNext);
			IoDeleteDevice(pdevo);
			break;
#endif // !PGP_EROS

		case PGPUTIL_DEV:
			pgpUtilDev = pdevo;
			break;
		}

		pdevo = pdevoNext;
	}

	// now get rid of the utility driver device objects
	if (pgpUtilDev)
	{
		// Unlock all blocks referenced in list and free list
		if (KeGetCurrentIrql () < DISPATCH_LEVEL) 
		{
			pdeve = pgpUtilDev->DeviceExtension;
			pgpMemlockCleanup (&pdeve->pdrve->memlock, 
				&pdeve->pdrve->csMemlock);
		}

		// Delete the symbolic link
		RtlInitUnicodeString (&deviceLinkUniString,
							  deviceLinkBuffer);
		IoDeleteSymbolicLink (&deviceLinkUniString);

		// Delete the device object
		PGPdbgVerbosePrint (("PGPutil: unloading.\n"));
		IoDeleteDevice(pgpUtilDev);
	}
}


#if !PGP_EROS
//	______________________________________________________
//
//	This function creates a filter Device object
//	and attaches it to the specified target.

static NTSTATUS
sAttachFilter (
	IN PDRIVER_OBJECT 		pdrvoFilter,
	IN PDEVICE_OBJECT		pdevoTarget,
	IN PDRVEXTENSION		pdrve,
	IN ULONG				ulNTDevType, 
	IN ULONG				ulPrivDevType)
{
	PDEVICE_OBJECT	pdevoFilter;
	PDEVICE_OBJECT	pdevoNextTarget;
	PDEVEXTENSION	pdeveFilter;
	NTSTATUS 		ntstatus;

	// Create a Device object without any name...
	ntstatus = IoCreateDevice (pdrvoFilter, sizeof(DEVEXTENSION),
				NULL, ulNTDevType, 0, FALSE, &pdevoFilter);

	if (!NT_SUCCESS (ntstatus))
	{
		PGPdbgPrint (("PGPutil: sAttachFilter: "
						"IoCreateDevice failed.\n"));
		return ntstatus;
	}

	// Attach to lower-level device.
	pdevoNextTarget = IoAttachDeviceToDeviceStack (pdevoFilter, pdevoTarget);

	if (!pdevoNextTarget)
	{
		PGPdbgPrint (("PGPutil: sAttachFilter: "
						"IoAttachDeviceToDeviceStack failed.\n"));
		IoDeleteDevice (pdevoFilter);
		return STATUS_DEVICE_BUSY;
	}

	// Set up filter's Device Extension...
	pdeveFilter = pdevoFilter->DeviceExtension;
	pdeveFilter->ulDeviceType		= ulPrivDevType;
	pdeveFilter->pdrve				= pdrve;
	pdeveFilter->pdevoNext			= pdevoNextTarget;
	pdeveFilter->ulSkipCount		= 0;
	pdeveFilter->ucLogicalDrive		= 0;		// unused by this device
	pdeveFilter->bIsAnsi			= FALSE;	// unused by this device
	pdeveFilter->bIsNTFS			= FALSE;	// unused by this device

	// don't touch the following, it may look illogical for Win2k but this is
	// how Microsoft's sample does it and Win2k bugs if you don't do it like
	// this

#if (_WIN32_WINNT >= 0x0500)
	pdevoFilter->Flags |= (DO_BUFFERED_IO | DO_POWER_PAGABLE);
#else	//  (_WIN32_WINNT < 0x0500)
	pdevoFilter->Flags = pdevoTarget->Flags & (DO_BUFFERED_IO | DO_DIRECT_IO);
#endif	//  (_WIN32_WINNT >= 0x0500)
	
    pdevoFilter->Flags &= ~DO_DEVICE_INITIALIZING;

	return STATUS_SUCCESS;
}


//	______________________________________________________
//
//	called by the system once per second

static VOID
sTimerCallback (
    IN PDEVICE_OBJECT	pdevo,
    IN PVOID			context)
{
	PDRVEXTENSION	pdrve	= ((PDEVEXTENSION)context)->pdrve;

	pdrve->ulStatusFlags |= kPGPUDFlag_InactivityTimerRunning;

	pgpProcessCacheTimerEvent (
			&pdrve->cacheSign, &pdrve->csCache);
	pgpProcessCacheTimerEvent (
			&pdrve->cacheDecrypt, &pdrve->csCache);
	pgpProcessInactivityTimerEvent (
			&pdrve->inactivity, &pdrve->csInactivity);
}


#if	(_WIN32_WINNT >= 0x0500)

//	______________________________________________________
//
// PGPUtilAddDevice is only set and only called under Win2k

NTSTATUS 
PGPUtilAddDevice(
    IN PDRIVER_OBJECT   Driver,
    IN PDEVICE_OBJECT   PDO
    )
{
	NTSTATUS		ntstatus	= STATUS_NO_SUCH_DEVICE;
	PDEVEXTENSION	pdeve		= NULL;
	PDEVICE_OBJECT	pdevo		= NULL;
	PDEVICE_OBJECT	pdevoNext	= NULL;

	PGPdbgPrint (("PGPutil: Win2k AddDevice.\n"));

	// find PGPUtil device object
	pdevo = Driver->DeviceObject;

	while (pdevo)
	{
		pdevoNext = pdevo->NextDevice;
		pdeve = pdevo->DeviceExtension;

		if (pdeve->ulDeviceType == PGPUTIL_DEV)
			break;

		pdevo = pdevoNext;
	}

	if (pdevo != NULL)
	{
		ULONG	used;
		WCHAR	guid[64];

		// get GUID id of PDO
		ntstatus = IoGetDeviceProperty(PDO, DevicePropertyClassGuid, 
			sizeof(guid), guid, &used);

		if (NT_SUCCESS(ntstatus))
		{
			UNICODE_STRING	devUni, kbUni, mouseUni;

			RtlInitUnicodeString(&devUni, guid);
			RtlInitUnicodeString(&kbUni, keyboardGuid);
			RtlInitUnicodeString(&mouseUni, mouseGuid);

			// attach appropriate filter
			if (RtlEqualUnicodeString(&devUni, &kbUni, TRUE))
			{
				ntstatus = sAttachFilter(Driver, PDO, pdeve->pdrve, 
					FILE_DEVICE_KEYBOARD, KBD_ENTROPY_DEV);

				if (NT_SUCCESS (ntstatus))
				{
					pdeve->pdrve->ulStatusFlags |= 
							kPGPUDFlag_KeyboardHookInstalled;
				}
			}
			else if (RtlEqualUnicodeString(&devUni, &mouseUni, TRUE))
			{
				ntstatus = sAttachFilter(Driver, PDO, pdeve->pdrve, 
					FILE_DEVICE_MOUSE, MOUSE_ENTROPY_DEV);

				if (NT_SUCCESS (ntstatus))
				{
					pdeve->pdrve->ulStatusFlags |= 
							kPGPUDFlag_MouseHookInstalled;
				}
			}
		}
	}

	// Bill sez return STATUS_SUCCESS in all cases
	return STATUS_SUCCESS;
}

#else	// (_WIN32_WINNT < 0x0500)

//	______________________________________________________
//
// PGPUtilNT4HookInput is called during DriverEntry for NT4 only

NTSTATUS 
PGPUtilNT4HookInput(PDRIVER_OBJECT pdrvo, PDEVEXTENSION pdeve)
{
	NTSTATUS		ntstatus	= STATUS_SUCCESS;
	PDEVICE_OBJECT 	pdevoTarget	= NULL;
	PFILE_OBJECT	pfoKBD		= NULL;
	PFILE_OBJECT 	pfoMouse	= NULL;

	UNICODE_STRING 	szMouseDeviceName;
	UNICODE_STRING 	szKbdDeviceName;

	RtlInitUnicodeString (&szMouseDeviceName, MOUSE_DEVICE_NAME);
	ntstatus = IoGetDeviceObjectPointer (&szMouseDeviceName,
					FILE_ALL_ACCESS, &pfoMouse, &pdevoTarget);

	if (!NT_SUCCESS (ntstatus))
	{
		PGPdbgPrint (("PGPutil: DriverEntry: "
						"mouse IoGetDeviceObjectPointer failed.\n"));
		return ntstatus;
	}

	// Try to attached our own filter device object to the device object 
	ntstatus = STATUS_NO_SUCH_DEVICE;
	while ((pdevoTarget != NULL) &&
		   (!NT_SUCCESS (ntstatus)))
	{
		ntstatus = sAttachFilter (pdrvo, pdevoTarget, pdeve->pdrve, 
			FILE_DEVICE_MOUSE, MOUSE_ENTROPY_DEV);
		pdevoTarget = pdevoTarget->NextDevice;
	}

	// decrement the ref count to the unused file object
	if (pfoMouse)
		ObDereferenceObject (pfoMouse);

	// indicate success
	pdeve->pdrve->ulStatusFlags |= kPGPUDFlag_MouseHookInstalled;


	// Now handle stuff for the keyboard entropy driver
	RtlInitUnicodeString (&szKbdDeviceName, KBD_DEVICE_NAME);
	ntstatus = IoGetDeviceObjectPointer (&szKbdDeviceName,
					FILE_ALL_ACCESS, &pfoKBD, &pdevoTarget);

	if (!NT_SUCCESS (ntstatus))
	{
		PGPdbgPrint (("PGPutil: DriverEntry: "
						"keyboard IoGetDeviceObjectPointer failed.\n"));
		return ntstatus;
	}

	// Try to attached our own filter device object to the device object
	ntstatus = STATUS_NO_SUCH_DEVICE;
	while ((pdevoTarget != NULL) &&
		   (!NT_SUCCESS (ntstatus)))
	{
		ntstatus = sAttachFilter (pdrvo, pdevoTarget, pdeve->pdrve, 
			FILE_DEVICE_KEYBOARD, KBD_ENTROPY_DEV);
		pdevoTarget = pdevoTarget->NextDevice;
	}

	// decrement the ref count to the unused file object
	if (pfoKBD)
		ObDereferenceObject (pfoKBD);

	// indicate success
	pdeve->pdrve->ulStatusFlags |= kPGPUDFlag_KeyboardHookInstalled;

	return ntstatus;
}

#endif	// (_WIN32_WINNT >= 0x0500)
#endif // !PGP_EROS

//	______________________________________________________
//
//  called when this driver is being initialized
	
NTSTATUS
DriverEntry(
    IN PDRIVER_OBJECT  pdrvo,
    IN PUNICODE_STRING RegistryPath)
{
    NTSTATUS		ntstatus	= STATUS_SUCCESS;
	PDEVEXTENSION	pdeve		= NULL;
	PDEVICE_OBJECT	pgpUtilDevo	= NULL;

	UNICODE_STRING	deviceNameUniString;
	UNICODE_STRING	deviceLinkUniString;
	ULONG			u;

	PGPdbgVerbosePrint (("PGPutil: entering DriverEntry.\n"));

#if !PGP_EROS
#if	(_WIN32_WINNT >= 0x0500)
	pdrvo->DriverExtension->AddDevice = PGPUtilAddDevice;	
#endif	// (_WIN32_WINNT >= 0x0500)
#endif // !PGP_EROS

	// Create an NON exclusive device object (multiple threads
	// can make requests to this device at a time)
	RtlInitUnicodeString (&deviceNameUniString, deviceNameBuffer);

	ntstatus = IoCreateDevice (pdrvo,
								sizeof(DEVEXTENSION),
								&deviceNameUniString,
								FILE_DEVICE_PGPUTILITY,
								0,
								FALSE,
								&pgpUtilDevo);

	if (NT_SUCCESS(ntstatus))
	{
		for (u=0; u<=IRP_MJ_MAXIMUM_FUNCTION; u++)
			pdrvo->MajorFunction[u] = sDispatch;

#if PGP_EROS
		// we don't support unloading of the driver when we are doing
		// keyboard/mouse filtering
		pdrvo->DriverUnload	= sUnload;
#endif // PGP_EROS

		// Create a symbolic link, e.g. a name that a Win32 app can specify
		// to open the device
		RtlInitUnicodeString (&deviceLinkUniString, deviceLinkBuffer);
		ntstatus = IoCreateSymbolicLink (&deviceLinkUniString,
											&deviceNameUniString);

		if (!NT_SUCCESS(ntstatus))
		{
			// Symbolic link creation failed- note this & then delete the
			// device object (it's useless if a Win32 app can't get at it).
			PGPdbgPrint (("PGPutil: IoCreateSymbolicLink failed.\n"));
			goto error;
		}
		else 
		{
			// initialize the utility device extension data structure
			pdeve = (PDEVEXTENSION)(pgpUtilDevo->DeviceExtension);
			memset (pdeve, 0, sizeof(DEVEXTENSION));
			pdeve->ulDeviceType = PGPUTIL_DEV;

			pdeve->pdrve = pgpDriverSecureAlloc (sizeof(DRVEXTENSION));
			if (pdeve->pdrve)
			{
				memset (pdeve->pdrve, 0, sizeof(DRVEXTENSION));

				pgpDriverInitCriticalSection (&pdeve->pdrve->csMemlock);
				if (pgpMemlockInit (&pdeve->pdrve->memlock))
				{
					pdeve->pdrve->ulStatusFlags |= 
							kPGPUDFlag_MemlockInitialized;
				}
				else
				{
					PGPdbgPrint (("PGPutil: pgpMemlockInit failed.\n"));
				}

#if !PGP_EROS
				pgpDriverInitCriticalSection (&pdeve->pdrve->csEntropy);
				pgpEntropyInit ();

				pgpDriverInitCriticalSection (&pdeve->pdrve->csCache);
				pgpInitCache (&pdeve->pdrve->cacheSign);
				pgpInitCache (&pdeve->pdrve->cacheDecrypt);

				pgpDriverInitCriticalSection (&pdeve->pdrve->csInactivity);
				pgpInitInactivity (&pdeve->pdrve->inactivity);

				ntstatus = pgpWipeDeleteInit (pdrvo, pdeve->pdrve);
				if (NT_SUCCESS (ntstatus))
				{
					pdeve->pdrve->ulStatusFlags |= 
							kPGPUDFlag_WipeDeleteInitialized;
				}
				else
				{
					PGPdbgPrint (("PGPutil: pgpWipeDeleteInit failed.\n"));
				}
#endif // !PGP_EROS
			}
			else
			{
				PGPdbgPrint (("PGPutil: Err: DRVEXTENSION allocation failed.\n"));
				ntstatus = STATUS_NO_MEMORY;
				goto error;
			}
		}
	}
	else
	{
		PGPdbgPrint (("PGPutil: Err: IoCreateDevice failed.\n"));
		goto error;
	}

#if !PGP_EROS
#if	(_WIN32_WINNT < 0x0500)
	// in NT4 we attach to keyboard and mouse right now
	ntstatus = PGPUtilNT4HookInput(pdrvo, pdeve);
#endif	// (_WIN32_WINNT < 0x0500)

	// initialize the timer callback
	IoInitializeTimer (pgpUtilDevo, sTimerCallback, pdeve);
	IoStartTimer (pgpUtilDevo);
#endif // !PGP_EROS

    return STATUS_SUCCESS;

error:
	if (pgpUtilDevo)
	    sUnload (pdrvo);

#if !PGP_EROS && (_WIN32_WINNT >= 0x0500)
	// to prevent mouse and keyboard from being blocked if 
	// driver fails
	return STATUS_SUCCESS;
#else
	return ntstatus;
#endif
}
